<?php
/*
	Plugin Name: Boost Content Remote Poster
	Plugin URI: http://www.boostcontent.com/
	Description: Open up for Boost Content remote posting.
	Author: Boost Content Team
	Version: 1.0
	Author URI: http://www.boostcontent.com/
	Text Domain: boostcontent
*/

class BoostContentRemotePoster {

	private $endPoint = "https://www.boostcontent.com/api";
	private $optionPrefix = "BoostContentRemotePoster_";
	private $tokenTTL = 300;

	/* Your Boost Content API key goes here. Or you can set it in admin. */
	private static $apiKey = "";

	public static function initialize() {
		if (is_admin()) return;

		if (self::$apiKey == "") {
			if (get_option('boostContentApiKey') == "") return;
			self::$apiKey = get_option('boostContentApiKey');
		}

		$remotePoster = new BoostContentRemotePoster();
		$remotePoster->handleRequest();
	}

	public static function createMenu() {
		add_menu_page('BoostContent Settings', 'BoostContent', 'administrator', __FILE__, array('BoostContentRemotePoster', 'settingsPage'));

		add_action('admin_init', array('BoostContentRemotePoster', 'registerSettings'));
	}

	public static function registerSettings() {
		register_setting('boostcontent', 'boostContentApiKey');
	}

	public static function generateHead() {
		if (is_single()) {
			global $wp_query;
			
			$metaTitle = get_post_meta(get_the_ID(), 'boostcontent_meta_title', true);
			if ($metaTitle) {
				printf('<meta name="title" content="%s" />', htmlentities($metaTitle));
			}

			$metaDescription = get_post_meta(get_the_ID(), 'boostcontent_meta_description', true);
			if ($metaDescription) {
				printf('<meta name="description" content="%s" />', htmlentities($metaDescription));
			}
			wp_reset_query();
			wp_reset_postdata();
		}
	}

	public static function settingsPage() {
		?>
		<div class="wrap">
		<h2>BoostContent</h2>

		<form method="post" action="options.php">
			<?php settings_fields( 'boostcontent' ); ?>
			<?php do_settings_sections( 'boostcontent' ); ?>
			<table class="form-table">
				<tr valign="top">
				<th scope="row">BoostContent API Key</th>
				<td><input type="text" name="boostContentApiKey" value="<?php echo esc_attr( self::$apiKey != "" ? self::$apiKey : get_option('boostContentApiKey') ); ?>" <?php echo self::$apiKey != "" ? ' disabled' : '';?> /></td>
				</tr>
			</table>
			
			<?php submit_button(); ?>

		</form>
		</div>
		<?php
	}

	private function handleRequest() {
		if (!isset($_POST['BoostContentRemotePoster'])) return;
		if (!isset($_POST['method'])) {
			$this->error('Missing parameter method');
		}

		$method = $_POST['method'];

		if ($method === "getToken") {
			$this->getToken(self::$apiKey);
			exit(0);
		}

		if (!$this->authenticate()) {
			return;
		}

		if ($method === "addPost") {
			$request = json_decode(base64_decode($_POST['request']));
			if ($request !== null) $this->addPost($request);
		} elseif ($method === "getPost") {
			$request = json_decode(base64_decode($_POST['request']));
			if ($request !== null) $this->getPost($request);
		} elseif ($method === "updatePost") {
			$request = json_decode(base64_decode($_POST['request']));
			if ($request !== null) $this->updatePost($request);
		}

		exit(0);
	}

	private function getToken($apiKey) {
		$header = "Content-Type: application/x-www-form-urlencoded";
		$data = array(
			'apiKey' => $apiKey,
			'asToken' => 1
		);
		$data = http_build_query($data);
		$context = array(
			'http' => array(
				'method' => 'POST',
				'header' => $header,
				'content' => $data
			)
		);
		$context = stream_context_create($context);

		$result = file_get_contents($this->endPoint.'/getToken', false, $context);

		if ($result === false) {
			$this->error('Could not fetch token');
		}

		$result = json_decode($result);
		if ($result === null or !isset($result->response)) {
			$this->error('Invalid token response');
		}

		$token = $result->response->token;

		// Save token.
		update_option($this->optionPrefix.'Token', $token);
		update_option($this->optionPrefix.'TokenTimestamp', time());

		$this->success();
	}

	private function findAuthorId() {
		$users = get_users(array('orderby' => 'id'));

		$defaultAuthorId = get_option("wpdeau_global_default_author", true);

		if ($defaultAuthorId) {
			foreach ($users as $user) {
				if ($user->ID == $defaultAuthorId) {
					return $user->ID;
				}
			}
		}

		foreach ($users as $user) {
			return $user->ID;
		}

		return NULL;
	}

	private function addPost($post) {

		if (!isset($post->postTitle)) return;
		if (!isset($post->postContent)) return;
		if (!isset($post->postDate)) return;

		$postAuthorId = $this->findAuthorId();

		if ($postAuthorId === null) {
			$this->error('Could not find user');
		}

		if (isset($post->postAuthorId)) {
			$postAuthorId = $post->postAuthorId;
		}

		$attachmentIds = [];
		if (isset($post->postImages)) {
			require_once(ABSPATH . "wp-admin" . '/includes/image.php');
			$uploadDir = wp_upload_dir();
			if (!is_writeable($uploadDir['path'])) {
				$this->error("Upload path is not writable");
			}
			foreach ($post->postImages as $image) {
				$imageData = file_get_contents($image->url);
				$filePath = $uploadDir['path'] . '/' . $image->name;
				file_put_contents($filePath, $imageData);

				$attachment = array(
					'guid' => $filePath,
					'post_mime_type' => $image->type,
					'post_title' => $image->title,
					'post_content' => '',
					'post_status' => 'inherit'
				);

				$attachmentId = wp_insert_attachment($attachment, $filePath);
				$attachmentData = wp_generate_attachment_metadata($attachmentId, $filePath);
				wp_update_attachment_metadata($attachmentId,  $attachmentData);
				$attachmentIds[] = $attachmentId;
			}
		}

		$postData = array(
			'post_status' => isset($post->postStatus) ? $post->postStatus : 'draft',
			'post_author' => $postAuthorId,
			'post_title' => $post->postTitle,
			'post_content' => $post->postContent,
			'post_date' => $post->postDate,
			'post_date_gmt' => $post->postDate,
			'post_type' => 'post'
		);

		if (isset($post->postCategory)) {
			if ($term = term_exists($post->postCategory, "category")) {
			} else {
				$term = wp_insert_term($post->postCategory, "category");
			}
			$postData['post_category'] = array($term['term_id']);
		}

		if (function_exists('kses_remove_filters')) kses_remove_filters();

		if (($postId = wp_insert_post($postData)) === 0) {
			$this->error('Could not insert post');
		}
		if (count($attachmentIds) > 0) {
			set_post_thumbnail($postId, $attachmentIds[0]);
		}
		if (isset($post->metaTitle) && isset($post->metaDescription) && $post->metaTitle && $post->metaDescription) {
			update_post_meta($postId, 'boostcontent_meta_title', $post->metaTitle);
			update_post_meta($postId, 'boostcontent_meta_description', $post->metaDescription);
		}
		if (function_exists('kses_init_filters')) kses_init_filters();
		$this->success(array(
			'postId' => $postId
		));
	}

	private function getPost($post) {

		if (!isset($post->postId)) return;

		$wpPost = get_post($post->postId);

		$this->success($wpPost);
	}

	private function updatePost($post) {

		if (!isset($post->postTitle)) return;
		if (!isset($post->postContent)) return;
		if (!isset($post->postDate)) return;
		if (!isset($post->postAuthor)) return;

		$postData = array(
			'post_status' => isset($post->postStatus) ? $post->postStatus : 'draft',
			'post_title' => $post->postTitle,
			'post_content' => $post->postContent,
			'post_date' => $post->postDate,
			'post_date_gmt' => $post->postDate,
			'post_type' => 'post'
		);

		if (isset($post->postCategory)) {
			if ($term = term_exists($post->postCategory, "category")) {
			} else {
				$term = wp_insert_term($post->postCategory, "category");
			}
			$postData['post_category'] = array($term['term_id']);
		}

		if (function_exists('kses_remove_filters')) kses_remove_filters();

		if (($postId = wp_update_post($postData)) === 0) {
			$this->error('Could not update post');
		}
		
		if (function_exists('kses_init_filters')) kses_init_filters();

		$this->success(array(
			'postId' => $postId
		));
	}

	private function authenticate() {

		if (!isset($_POST['token']) or !isset($_POST['request']) or !isset($_POST['hmac'])) {
			return false;
		}
		
		$token = get_option($this->optionPrefix.'Token');
		$timestamp = get_option($this->optionPrefix.'TokenTimestamp');

		if ($token === false or $timestamp === false) {
			return false;
		}

		if ($timestamp+$this->tokenTTL < time()) {
			return false;
		}

		if ($_POST['token'] != $token) {
			return false;
		}

		// Validate hmac.
		if (hash_hmac('sha256', $_POST['token'].$_POST['request'], $token) !== $_POST['hmac']) {
			return false;
		}

		update_option($this->optionPrefix.'Token', false);
		update_option($this->optionPrefix.'TokenTimestamp', false);

		return true;
	}

	private function error($response) {
		header("HTTP/1.0 500 Internal Server Error");
		echo json_encode(array(
			'status' => 'error',
			'response' => $response
		));
		exit(0);
	}

	private function success($response = array()) {
		echo json_encode(array(
			'status' => 'success',
			'response' => $response
		));
		exit(0);
	}

}

if (function_exists('add_action')) {
	add_action('admin_menu', array('BoostContentRemotePoster', 'createMenu'));
	add_action('init', array('BoostContentRemotePoster', 'initialize'));
	add_action('wp_head', array('BoostContentRemotePoster', 'generateHead'));
}

?>
